from flask_openapi3 import APIBlueprint, Tag
from pydantic import BaseModel
from app.models import User, db
from app.schemas import UserCreate, UserUpdate, UserResponse, PaginatedResponse
from app.decorators import token_required
from sqlalchemy.exc import IntegrityError
from app.config import Config
from app.errors import api_response, success_response, error_response, ErrorDetail, SuccessResponse, ErrorResponse

user_tag = Tag(name="Users", description="Operações com usuários")
user_bp = APIBlueprint("users", __name__, url_prefix="/api/users", abp_tags=[user_tag])

class QueryParams(BaseModel):
    page: int = 1
    per_page: int = Config.ITEMS_PER_PAGE

@user_bp.get(
    "/",
    responses={
        "200": SuccessResponse,
        "401": ErrorResponse
    },
    security=[{"bearerAuth": []}]
)
@token_required
@api_response
def list_users(query: QueryParams):
    """Lista todos os usuários (paginado)"""
    pagination = User.query.paginate(page=query.page, per_page=query.per_page)
    
    items = []
    for user in pagination.items:
        user_dict = {
            "id": user.id,
            "username": user.username,
            "email": user.email,
            "created_at": user.created_at_str,
            "updated_at": user.updated_at_str
        }
        items.append(user_dict)
    
    data = {
        "items": items,
        "total": pagination.total,
        "page": query.page,
        "per_page": query.per_page,
        "total_pages": pagination.pages
    }
    
    return success_response(
        message="Usuários listados com sucesso",
        data=data,
        code=200
    )

class UserIdPath(BaseModel):
    user_id: int

@user_bp.get(
    "/<int:user_id>",
    responses={
        "200": SuccessResponse,
        "401": ErrorResponse,
        "404": ErrorResponse
    },
    security=[{"bearerAuth": []}]
)
@token_required
@api_response
def get_user(path: UserIdPath):
    """Obtém um usuário específico"""
    user = User.query.get_or_404(path.user_id)
    
    user_data = {
        "id": user.id,
        "username": user.username,
        "email": user.email,
        "first_name": user.first_name,
        "last_name": user.last_name,
        "phone": user.phone,
        "birth_date": user.birth_date,
        "created_at": user.created_at_str,
        "updated_at": user.updated_at_str
    }
    
    return success_response(
        message="Usuário encontrado com sucesso",
        data=user_data,
        code=200
    )

@user_bp.patch(
    "/<int:user_id>",
    responses={
        "200": SuccessResponse,
        "400": ErrorResponse,
        "401": ErrorResponse,
        "403": ErrorResponse,
        "404": ErrorResponse
    },
    security=[{"bearerAuth": []}]
)
@token_required
@api_response
def update_user(path: UserIdPath, body: UserUpdate):
    """Atualiza um usuário"""
    user = User.query.get_or_404(path.user_id)
    
    if not hasattr(update_user, 'current_user') or update_user.current_user.id != path.user_id:
        return error_response(
            message="Não autorizado para atualizar este usuário",
            errors=[ErrorDetail(field="authorization", message="Usuário não tem permissão")],
            code=403
        )
    
    try:
        if body.username is not None:
            user.username = body.username
        if body.email is not None:
            user.email = body.email
        if body.password is not None:
            user.set_password(body.password)
        
        db.session.commit()
        
        user_data = {
            "id": user.id,
            "username": user.username,
            "email": user.email,
            "first_name": user.first_name,
            "last_name": user.last_name,
            "phone": user.phone,
            "birth_date": user.birth_date,
            "created_at": user.created_at_str,
            "updated_at": user.updated_at_str
        }
        
        return success_response(
            message="Usuário atualizado com sucesso",
            data=user_data,
            code=200
        )
        
    except IntegrityError:
        db.session.rollback()
        return error_response(
            message="Erro ao atualizar usuário",
            errors=[ErrorDetail(field="unique", message="Usuário ou email já existe")],
            code=400
        )

@user_bp.delete(
    "/<int:user_id>",
    responses={
        "200": SuccessResponse,
        "401": ErrorResponse,
        "403": ErrorResponse,
        "404": ErrorResponse
    },
    security=[{"bearerAuth": []}]
)
@token_required
@api_response
def delete_user(path: UserIdPath):
    """Deleta um usuário"""
    if not hasattr(delete_user, 'current_user') or delete_user.current_user.id != path.user_id:
        return error_response(
            message="Não autorizado para deletar este usuário",
            errors=[ErrorDetail(field="authorization", message="Usuário não tem permissão")],
            code=403
        )
    
    user = User.query.get_or_404(path.user_id)
    
    try:
        db.session.delete(user)
        db.session.commit()
        
        return success_response(
            message="Usuário deletado com sucesso",
            code=200
        )
        
    except IntegrityError:
        db.session.rollback()
        return error_response(
            message="Não é possível deletar este usuário",
            errors=[ErrorDetail(field="foreign_key", message="Existem registros vinculados a este usuário")],
            code=400
        )