from enum import Enum
from pydantic import BaseModel, EmailStr, Field, StringConstraints
from typing import Optional, Annotated, List
from datetime import date
from decimal import Decimal
from typing_extensions import Annotated

# Schema base para respostas
class BaseResponseSchema(BaseModel):
    """Schema base para todas as respostas que incluem timestamps"""
    request_time: float = Field(default=0.0, description="Tempo de processamento da requisição em segundos")

    class Config:
        from_attributes = True

        @classmethod
        def from_orm(cls, obj):
            # Converte datetime para string ISO
            if hasattr(obj, 'created_at') and obj.created_at:
                obj.created_at = obj.created_at.isoformat()
            if hasattr(obj, 'updated_at') and obj.updated_at:
                obj.updated_at = obj.updated_at.isoformat()
            return super().from_orm(obj)

class MessageResponse(BaseResponseSchema):
    """Schema base para respostas com mensagem"""
    message: str
    request_time: float = 0.0

# Schemas base
class UserBase(BaseModel):
    """Schema base para usuários"""
    username: Annotated[str, StringConstraints(min_length=3, max_length=80)] = Field(
        description="Nome de usuário (entre 3 e 80 caracteres)",
        examples=["john_doe", "maria.silva"]
    )
    email: EmailStr = Field(
        description="Endereço de e-mail válido",
        examples=["usuario@exemplo.com"]
    )

class CategoryType(str, Enum):
    """Tipos de categoria disponíveis"""
    EXPENSE = "expense"  # Despesa
    INCOME = "income"   # Receita

class CategoryBase(BaseModel):
    """Schema base para categorias de transações"""
    name: Annotated[str, StringConstraints(min_length=1, max_length=50)] = Field(
        description="Nome da categoria",
        examples=["Alimentação", "Moradia", "Transporte", "Salário"])
    type: CategoryType = Field(
        description="Tipo da categoria (expense para despesas, income para receitas)")
    is_fixed: bool = Field(
        default=False,
        description="Indica se é uma categoria de valor fixo (ex: aluguel, salário)",
        examples=[True, False])
    description: Optional[str] = Field(
        None,
        description="Descrição detalhada da categoria",
        examples=["Gastos com alimentação", "Despesas com transporte público", "Salário mensal"])

class TransactionBase(BaseModel):
    """Schema base para transações (despesas e receitas)"""
    description: Annotated[str, StringConstraints(min_length=1, max_length=200)] = Field(
        description="Descrição da transação",
        examples=["Almoço", "Conta de luz", "Pagamento do salário"]
    )
    amount: Decimal = Field(
        gt=0,
        description="Valor da transação (sempre positivo)",
        examples=[29.90, 150.00, 3500.00]
    )
    date: str = Field(
        description="Data da transação (formato YYYY-MM-DD)",
        examples=["2025-10-10"],
        pattern=r'^\d{4}-\d{2}-\d{2}$'
    )
    is_recurring: bool = Field(
        default=False,
        description="Indica se a transação é recorrente",
        examples=[True, False]
    )
    recurrence_end_date: Optional[str] = Field(
        default=None,
        description="Data final da recorrência (formato YYYY-MM-DD), obrigatório se is_recurring=True",
        examples=["2026-12-31"],
        pattern=r'^\d{4}-\d{2}-\d{2}$'
    )
    category_id: int = Field(
        description="ID da categoria associada à transação",
        examples=[1, 2, 3]
    )

# Schemas para cria��o
class UserCreate(UserBase):
    password: Annotated[str, StringConstraints(min_length=6)]

class CategoryCreate(CategoryBase):
    pass

class ExpenseCreate(TransactionBase):
    pass

class IncomeCreate(TransactionBase):
    pass

# Schemas para atualiza��o
class UserUpdate(BaseModel):
    username: Optional[Annotated[str, StringConstraints(min_length=3, max_length=80)]] = None
    email: Optional[EmailStr] = None
    password: Optional[Annotated[str, StringConstraints(min_length=6)]] = None

class CategoryUpdate(BaseModel):
    name: Optional[Annotated[str, StringConstraints(min_length=1, max_length=50)]] = None
    description: Optional[str] = None
    type: Optional[Annotated[str, StringConstraints(pattern="^(expense|income)$")]] = None

class TransactionPatch(BaseModel):
    """Schema para atualização parcial de transações"""
    description: Optional[Annotated[str, StringConstraints(min_length=1, max_length=200)]] = Field(
        default=None,
        description="Nova descrição da transação",
        examples=["Almoço", "Conta de luz", "Pagamento do salário"]
    )
    amount: Optional[Decimal] = Field(
        default=None,
        gt=0,
        description="Novo valor da transação (sempre positivo)",
        examples=[29.90, 150.00, 3500.00]
    )
    date: Optional[str] = Field(
        default=None,
        description="Nova data da transação (formato YYYY-MM-DD)",
        examples=["2025-10-10"],
        pattern=r'^\d{4}-\d{2}-\d{2}$'
    )
    is_recurring: Optional[bool] = Field(
        default=None,
        description="Indica se a transação é recorrente",
        examples=[True, False]
    )
    recurrence_end_date: Optional[str] = Field(
        default=None,
        description="Nova data final da recorrência (formato YYYY-MM-DD), obrigatório se is_recurring=True",
        examples=["2026-12-31"],
        pattern=r'^\d{4}-\d{2}-\d{2}$'
    )
    category_id: Optional[int] = Field(
        default=None,
        description="Novo ID da categoria associada à transação",
        examples=[1, 2, 3]
    )

# Alias para manter compatibilidade com código existente
TransactionUpdate = TransactionPatch

# Schemas para resposta
class UserResponse(UserBase, BaseResponseSchema):
    """Schema de resposta para usuários"""
    id: int = Field(description="ID único do usuário")
    created_at: str = Field(description="Data e hora de criação (formato ISO)")
    updated_at: str = Field(description="Data e hora da última atualização (formato ISO)")

from datetime import datetime

class CategoryResponse(CategoryBase, BaseResponseSchema):
    """Schema de resposta para categorias"""
    id: int = Field(description="ID único da categoria")
    user_id: int = Field(description="ID do usuário dono da categoria")
    created_at: str = Field(description="Data e hora de criação (formato ISO)")
    updated_at: str = Field(description="Data e hora da última atualização (formato ISO)")

class ExpenseResponse(TransactionBase, BaseResponseSchema):
    """Schema de resposta para despesas"""
    id: int = Field(description="ID único da despesa")
    user_id: int = Field(description="ID do usuário dono da despesa")
    category: CategoryResponse = Field(description="Dados da categoria associada")
    created_at: str = Field(description="Data e hora de criação (formato ISO)")
    updated_at: str = Field(description="Data e hora da última atualização (formato ISO)")

class IncomeResponse(TransactionBase, BaseResponseSchema):
    """Schema de resposta para receitas"""
    id: int = Field(description="ID único da receita")
    user_id: int = Field(description="ID do usuário dono da receita")
    category: CategoryResponse = Field(description="Dados da categoria associada")
    created_at: str = Field(description="Data e hora de criação (formato ISO)")
    updated_at: str = Field(description="Data e hora da última atualização (formato ISO)")

# Schemas para paginação
class PaginatedResponse(BaseResponseSchema):
    """Schema para respostas paginadas"""
    items: List = Field(description="Lista de itens da página atual")
    total: int = Field(description="Total de itens em todas as páginas")
    page: int = Field(description="Número da página atual")
    per_page: int = Field(description="Quantidade de itens por página")
    total_pages: int = Field(description="Total de páginas disponíveis")

# Schema para autenticação
class LoginSchema(BaseModel):
    """
    Schema para autenticação de usuário
    
    Headers necessários:
    - Content-Type: application/json
    """
    username: str = Field(
        description="Nome de usuário",
        examples=["john_doe"]
    )
    password: str = Field(
        description="Senha do usuário",
        examples=["senha123"]
    )

class TokenResponse(BaseModel):
    access_token: str
    token_type: str = "bearer"
    request_time: float = 0.0
