from typing import Any, List, Optional
from pydantic import BaseModel, Field
from flask import jsonify, Response
from functools import wraps

class ErrorDetail(BaseModel):
    """Detalhes de um erro específico"""
    field: str = Field(..., description="Campo com erro")
    message: str = Field(..., description="Mensagem do erro")

class BaseAPIResponse(BaseModel):
    """Base para respostas da API"""
    status: str = Field(..., description="Status da resposta (success, error)")
    message: str = Field(..., description="Mensagem explicativa")
    code: int = Field(..., description="Código HTTP da resposta")

class SuccessResponse(BaseAPIResponse):
    """Resposta para requisições bem-sucedidas"""
    data: Optional[Any] = Field(None, description="Dados da resposta")

class ErrorResponse(BaseAPIResponse):
    """Resposta para erros"""
    errors: List[ErrorDetail] = Field(default_factory=list, description="Lista de erros")

def success_response(message: str, data: Any = None, code: int = 200) -> tuple:
    """Helper para criar respostas de sucesso"""
    response = SuccessResponse(
        status="success",
        message=message,
        code=code,
        data=data
    )
    return jsonify(response.model_dump(exclude_none=True)), code

def error_response(message: str, errors: List[ErrorDetail] = None, code: int = 400) -> tuple:
    """Helper para criar respostas de erro"""
    response = ErrorResponse(
        status="error",
        message=message,
        code=code,
        errors=errors or []
    )
    return jsonify(response.model_dump(exclude_none=True)), code

def handle_validation_error(error):
    """Handler global para erros de validação"""
    errors = []
    
    # Tratamento para erros do Pydantic
    if hasattr(error, 'errors'):
        if callable(error.errors):  # Pydantic v1
            for e in error.errors():
                errors.append(ErrorDetail(
                    field='.'.join(str(x) for x in e['loc']),
                    message=e['msg']
                ))
        else:  # Pydantic v2
            for e in error.errors:
                errors.append(ErrorDetail(
                    field='.'.join(str(x) for x in e['loc']),
                    message=e['msg']
                ))
                
    # Tratamento para erros gerais de validação
    elif hasattr(error, 'messages'):
        for field, messages in error.messages.items():
            if isinstance(messages, (list, tuple)):
                message = '; '.join(messages)
            else:
                message = str(messages)
            errors.append(ErrorDetail(
                field=field,
                message=message
            ))
    
    # Tratamento para outros tipos de erro
    else:
        errors.append(ErrorDetail(
            field="validation",
            message=str(error)
        ))
        
    return error_response(
        message="Erro de validação",
        errors=errors,
        code=400
    )

def handle_database_error(error):
    """Handler global para erros de banco de dados"""
    return error_response(
        message="Erro ao acessar o banco de dados",
        errors=[ErrorDetail(field="database", message=str(error))],
        code=500
    )

def api_response(f):
    """Decorator para padronizar respostas da API"""
    @wraps(f)
    def decorated(*args, **kwargs):
        try:
            response = f(*args, **kwargs)
            
            # Se a resposta já é uma tupla (response, code)
            if isinstance(response, tuple):
                if isinstance(response[0], Response):
                    return response
                response_data, code = response
            else:
                response_data = response
                code = 200
                
            # Se a resposta já é uma Response do Flask
            if isinstance(response_data, Response):
                return response_data, code
            
            # Se é None, retorna sucesso sem dados
            if response_data is None:
                return success_response(
                    message="Operação realizada com sucesso",
                    code=code
                )
            
            # Se é um dicionário normal
            if isinstance(response_data, dict):
                return success_response(
                    message="Operação realizada com sucesso",
                    data=response_data,
                    code=code
                )
            
            # Para outros tipos de dados
            if not isinstance(response_data, (SuccessResponse, ErrorResponse)):
                response_data = {"data": response_data}
            
            return success_response(
                message="Operação realizada com sucesso",
                data=response_data,
                code=code
            )
            
        except ValueError as e:
            return error_response(
                message="Erro de validação",
                errors=[ErrorDetail(field="validation", message=str(e))],
                code=400
            )
        except Exception as e:
            return error_response(
                message="Erro interno do servidor",
                errors=[ErrorDetail(field="server", message=str(e))],
                code=500
            )
    return decorated