from flask_openapi3 import APIBlueprint, Tag
from pydantic import BaseModel
from app.models import User, db
from app.schemas import LoginSchema, TokenResponse
from app.schemas.user import UserCreateSchema
from app.decorators import generate_token
from sqlalchemy.exc import IntegrityError
from app.errors import api_response, success_response, error_response, ErrorDetail, SuccessResponse, ErrorResponse

auth_tag = Tag(name="Auth", description="Autenticação de usuários")
auth_bp = APIBlueprint("auth", __name__, url_prefix="/api/auth", abp_tags=[auth_tag])

@auth_bp.post(
    "/register",
    responses={
        "201": SuccessResponse,
        "400": ErrorResponse
    },
)
@api_response
def register(body: UserCreateSchema):
    """Registra um novo usuário"""
    errors = []
    # Validação de senha
    if body.password != body.confirm_password:
        errors.append(ErrorDetail(field="confirm_password", message="As senhas não conferem"))
    # Validação de campos obrigatórios
    if not body.username:
        errors.append(ErrorDetail(field="username", message="Campo obrigatório"))
    if not body.email:
        errors.append(ErrorDetail(field="email", message="Campo obrigatório"))
    if not body.password:
        errors.append(ErrorDetail(field="password", message="Campo obrigatório"))
    if errors:
        return error_response(
            message="Erro de validação",
            errors=errors,
            code=400
        )
        
    try:
        user = User(
            username=body.username,
            email=body.email,
            first_name=body.first_name,
            last_name=body.last_name,
            phone=body.phone,
            birth_date=body.birth_date
        )
        user.set_password(body.password)
        db.session.add(user)
        db.session.commit()
        
        user_data = {
            "id": user.id,
            "username": user.username,
            "email": user.email,
            "first_name": user.first_name,
            "last_name": user.last_name,
            "phone": user.phone,
            "birth_date": user.birth_date
        }
        
        return success_response(
            message="Usuário criado com sucesso",
            data=user_data,
            code=201
        )
        
    except IntegrityError:
        db.session.rollback()
        return error_response(
            message="E-mail ou usuário já cadastrado",
            errors=[ErrorDetail(field="email", message="E-mail ou usuário já cadastrado")],
            code=400
        )

@auth_bp.post(
    "/login",
    responses={
        "200": SuccessResponse,
        "401": ErrorResponse
    },
)
@api_response
def login(body: LoginSchema):
    """Login de usuário"""
    user = User.query.filter_by(username=body.username).first()
    
    if user and user.check_password(body.password):
        token = generate_token(user.id)
        return success_response(
            message="Login realizado com sucesso",
            data={"access_token": token},
            code=200
        )
    
    return error_response(
        message="Credenciais inválidas",
        errors=[ErrorDetail(field="credentials", message="Usuário ou senha inválidos")],
        code=401
    )