from flask_openapi3 import APIBlueprint, Tag
from pydantic import BaseModel
from app.models import Expense, Category, db
from app.schemas import (
    ExpenseCreate, TransactionUpdate, ExpenseResponse,
    PaginatedResponse, MessageResponse
)
from app.decorators import token_required
from app.config import Config
from app.errors import api_response, success_response, error_response, ErrorDetail, SuccessResponse, ErrorResponse
from datetime import datetime
from dateutil.relativedelta import relativedelta
from sqlalchemy import extract
from sqlalchemy.exc import IntegrityError

expense_tag = Tag(name="Expenses", description="Operações com despesas")
expense_bp = APIBlueprint("expenses", __name__, url_prefix="/api/expenses", abp_tags=[expense_tag])

class ExpenseIdPath(BaseModel):
    expense_id: int

@expense_bp.post(
    "/",
    responses={
        "201": SuccessResponse,
        "400": ErrorResponse,
        "401": ErrorResponse,
        "404": ErrorResponse
    },
    security=[{"bearerAuth": []}]
)
@token_required
@api_response
def create_expense(body: ExpenseCreate):
    """Cria uma nova despesa"""
    try:
        # Verifica se a categoria pertence ao usuário
        category = Category.query.filter_by(
            id=body.category_id,
            user_id=create_expense.current_user.id,
            type='expense'
        ).first_or_404()

        expense = Expense(
            description=body.description,
            amount=body.amount,
            date=body.date,
            is_recurring=body.is_recurring,
            recurrence_end_date=body.recurrence_end_date,
            category_id=category.id,
            user_id=create_expense.current_user.id
        )
        
        db.session.add(expense)
        expenses_created = [expense]
        
        # Se for recorrente, cria as despesas futuras até a data final
        if body.is_recurring and body.recurrence_end_date:
            current_date = body.date
            end_date = body.recurrence_end_date
            
            while current_date < end_date:
                current_date = current_date + relativedelta(months=1)
                recurring_expense = Expense(
                    description=body.description,
                    amount=body.amount,
                    date=current_date,
                    is_recurring=True,
                    recurrence_end_date=body.recurrence_end_date,
                    category_id=category.id,
                    user_id=create_expense.current_user.id
                )
                db.session.add(recurring_expense)
                expenses_created.append(recurring_expense)
        
        db.session.commit()
        
        expenses_data = []
        for exp in expenses_created:
            expenses_data.append({
                "id": exp.id,
                "description": exp.description,
                "amount": exp.amount,
                "date": exp.date.isoformat(),
                "is_recurring": exp.is_recurring,
                "recurrence_end_date": exp.recurrence_end_date.isoformat() if exp.recurrence_end_date else None,
                "category_id": exp.category_id,
                "user_id": exp.user_id,
                "created_at": exp.created_at_str,
                "updated_at": exp.updated_at_str
            })
        
        return success_response(
            message="Despesa(s) criada(s) com sucesso",
            data=expenses_data[0] if len(expenses_data) == 1 else expenses_data,
            code=201
        )
        
    except IntegrityError:
        db.session.rollback()
        return error_response(
            message="Erro ao criar despesa",
            errors=[ErrorDetail(field="integrity", message="Erro de integridade no banco de dados")],
            code=400
        )

class ExpenseQueryParams(BaseModel):
    page: int = 1
    per_page: int = Config.ITEMS_PER_PAGE
    month: int = None
    year: int = None
    category_id: int = None

@expense_bp.get(
    "/",
    responses={
        "200": SuccessResponse,
        "401": ErrorResponse,
    },
    security=[{"bearerAuth": []}]
)
@token_required
@api_response
def list_expenses(query: ExpenseQueryParams):
    """Lista todas as despesas do usuário (paginado)"""
    query_filter = Expense.query.filter_by(user_id=list_expenses.current_user.id)
    
    if query.month and query.year:
        query_filter = query_filter.filter(
            extract('month', Expense.date) == query.month,
            extract('year', Expense.date) == query.year
        )
    
    if query.category_id:
        query_filter = query_filter.filter_by(category_id=query.category_id)
    
    # Ordena por data mais recente
    query_filter = query_filter.order_by(Expense.date.desc())
    
    pagination = query_filter.paginate(page=query.page, per_page=query.per_page)
    
    items = []
    for expense in pagination.items:
        items.append({
            "id": expense.id,
            "description": expense.description,
            "amount": expense.amount,
            "date": expense.date.isoformat(),
            "is_recurring": expense.is_recurring,
            "recurrence_end_date": expense.recurrence_end_date.isoformat() if expense.recurrence_end_date else None,
            "category_id": expense.category_id,
            "user_id": expense.user_id,
            "created_at": expense.created_at_str,
            "updated_at": expense.updated_at_str
        })
    
    data = {
        "items": items,
        "total": pagination.total,
        "page": query.page,
        "per_page": query.per_page,
        "total_pages": pagination.pages
    }
    
    return success_response(
        message="Despesas listadas com sucesso",
        data=data,
        code=200
    )

@expense_bp.get(
    "/<int:expense_id>",
    responses={
        "200": SuccessResponse,
        "401": ErrorResponse,
        "404": ErrorResponse
    },
    security=[{"bearerAuth": []}]
)
@token_required
@api_response
def get_expense(path: ExpenseIdPath):
    """Obtém uma despesa específica"""
    expense = Expense.query.filter_by(
        id=path.expense_id,
        user_id=get_expense.current_user.id
    ).first_or_404()
    
    expense_data = {
        "id": expense.id,
        "description": expense.description,
        "amount": expense.amount,
        "date": expense.date.isoformat(),
        "is_recurring": expense.is_recurring,
        "recurrence_end_date": expense.recurrence_end_date.isoformat() if expense.recurrence_end_date else None,
        "category_id": expense.category_id,
        "user_id": expense.user_id,
        "created_at": expense.created_at_str,
        "updated_at": expense.updated_at_str
    }
    
    return success_response(
        message="Despesa encontrada com sucesso",
        data=expense_data,
        code=200
    )

@expense_bp.patch(
    "/<int:expense_id>",
    responses={
        "200": SuccessResponse,
        "400": ErrorResponse,
        "401": ErrorResponse,
        "404": ErrorResponse
    },
    security=[{"bearerAuth": []}]
)
@token_required
@api_response
def update_expense(path: ExpenseIdPath, body: TransactionUpdate):
    """Atualiza uma despesa"""
    try:
        expense = Expense.query.filter_by(
            id=path.expense_id,
            user_id=update_expense.current_user.id
        ).first_or_404()
        
        if body.category_id is not None:
            # Verifica se a nova categoria pertence ao usuário
            Category.query.filter_by(
                id=body.category_id,
                user_id=update_expense.current_user.id,
                type='expense'
            ).first_or_404()
            expense.category_id = body.category_id
        
        if body.description is not None:
            expense.description = body.description
        if body.amount is not None:
            expense.amount = body.amount
        if body.date is not None:
            expense.date = body.date
        if body.is_recurring is not None:
            expense.is_recurring = body.is_recurring
        if body.recurrence_end_date is not None:
            expense.recurrence_end_date = body.recurrence_end_date
        
        db.session.commit()
        
        expense_data = {
            "id": expense.id,
            "description": expense.description,
            "amount": expense.amount,
            "date": expense.date.isoformat(),
            "is_recurring": expense.is_recurring,
            "recurrence_end_date": expense.recurrence_end_date.isoformat() if expense.recurrence_end_date else None,
            "category_id": expense.category_id,
            "user_id": expense.user_id,
            "created_at": expense.created_at_str,
            "updated_at": expense.updated_at_str
        }
        
        return success_response(
            message="Despesa atualizada com sucesso",
            data=expense_data,
            code=200
        )
        
    except IntegrityError:
        db.session.rollback()
        return error_response(
            message="Erro ao atualizar despesa",
            errors=[ErrorDetail(field="integrity", message="Erro de integridade no banco de dados")],
            code=400
        )

class DeleteExpenseQueryParams(BaseModel):
    delete_recurring: bool = False

@expense_bp.delete(
    "/<int:expense_id>",
    responses={
        "200": SuccessResponse,
        "401": ErrorResponse,
        "404": ErrorResponse
    },
    security=[{"bearerAuth": []}]
)
@token_required
@api_response
def delete_expense(path: ExpenseIdPath, query: DeleteExpenseQueryParams):
    """Deleta uma despesa"""
    try:
        expense = Expense.query.filter_by(
            id=path.expense_id,
            user_id=delete_expense.current_user.id
        ).first_or_404()
        
        if query.delete_recurring and expense.is_recurring:
            # Deleta todas as despesas recorrentes futuras
            Expense.query.filter(
                Expense.user_id == delete_expense.current_user.id,
                Expense.description == expense.description,
                Expense.category_id == expense.category_id,
                Expense.is_recurring == True,
                Expense.date >= expense.date
            ).delete()
        else:
            db.session.delete(expense)
        
        db.session.commit()
        
        return success_response(
            message="Despesa(s) deletada(s) com sucesso",
            code=200
        )
        
    except IntegrityError:
        db.session.rollback()
        return error_response(
            message="Erro ao deletar despesa",
            errors=[ErrorDetail(field="integrity", message="Erro de integridade no banco de dados")],
            code=400
        )