from flask_openapi3 import APIBlueprint, Tag
from pydantic import BaseModel
from app.models import Income, Category, db
from app.schemas import (
    IncomeCreate, TransactionUpdate, IncomeResponse,
    PaginatedResponse, MessageResponse
)
from app.decorators import token_required
from app.config import Config
from app.errors import api_response, success_response, error_response, ErrorDetail, SuccessResponse, ErrorResponse
from datetime import datetime
from dateutil.relativedelta import relativedelta
from sqlalchemy import extract
from sqlalchemy.exc import IntegrityError

income_tag = Tag(name="Incomes", description="Operações com receitas")
income_bp = APIBlueprint("incomes", __name__, url_prefix="/api/incomes", abp_tags=[income_tag])

class IncomeIdPath(BaseModel):
    income_id: int

@income_bp.post(
    "/",
    responses={
        "201": SuccessResponse,
        "400": ErrorResponse,
        "401": ErrorResponse,
        "404": ErrorResponse
    },
    security=[{"bearerAuth": []}]
)
@token_required
@api_response
def create_income(body: IncomeCreate):
    """Cria uma nova receita"""
    try:
        # Verifica se a categoria pertence ao usuário
        category = Category.query.filter_by(
            id=body.category_id,
            user_id=create_income.current_user.id,
            type='income'
        ).first_or_404()

        income = Income(
            description=body.description,
            amount=body.amount,
            date=body.date,
            is_recurring=body.is_recurring,
            recurrence_end_date=body.recurrence_end_date,
            category_id=category.id,
            user_id=create_income.current_user.id
        )
        
        db.session.add(income)
        incomes_created = [income]
        
        # Se for recorrente, cria as receitas futuras até a data final
        if body.is_recurring and body.recurrence_end_date:
            current_date = body.date
            end_date = body.recurrence_end_date
            
            while current_date < end_date:
                current_date = current_date + relativedelta(months=1)
                recurring_income = Income(
                    description=body.description,
                    amount=body.amount,
                    date=current_date,
                    is_recurring=True,
                    recurrence_end_date=body.recurrence_end_date,
                    category_id=category.id,
                    user_id=create_income.current_user.id
                )
                db.session.add(recurring_income)
                incomes_created.append(recurring_income)
        
        db.session.commit()
        
        incomes_data = []
        for inc in incomes_created:
            incomes_data.append({
                "id": inc.id,
                "description": inc.description,
                "amount": inc.amount,
                "date": inc.date.isoformat(),
                "is_recurring": inc.is_recurring,
                "recurrence_end_date": inc.recurrence_end_date.isoformat() if inc.recurrence_end_date else None,
                "category_id": inc.category_id,
                "user_id": inc.user_id,
                "created_at": inc.created_at_str,
                "updated_at": inc.updated_at_str
            })
        
        return success_response(
            message="Receita(s) criada(s) com sucesso",
            data=incomes_data[0] if len(incomes_data) == 1 else incomes_data,
            code=201
        )
        
    except IntegrityError:
        db.session.rollback()
        return error_response(
            message="Erro ao criar receita",
            errors=[ErrorDetail(field="integrity", message="Erro de integridade no banco de dados")],
            code=400
        )

class IncomeQueryParams(BaseModel):
    page: int = 1
    per_page: int = Config.ITEMS_PER_PAGE
    month: int = None
    year: int = None
    category_id: int = None

@income_bp.get(
    "/",
    responses={
        "200": SuccessResponse,
        "401": ErrorResponse,
    },
    security=[{"bearerAuth": []}]
)
@token_required
@api_response
def list_incomes(query: IncomeQueryParams):
    """Lista todas as receitas do usuário (paginado)"""
    query_filter = Income.query.filter_by(user_id=list_incomes.current_user.id)
    
    if query.month and query.year:
        query_filter = query_filter.filter(
            extract('month', Income.date) == query.month,
            extract('year', Income.date) == query.year
        )
    
    if query.category_id:
        query_filter = query_filter.filter_by(category_id=query.category_id)
    
    # Ordena por data mais recente
    query_filter = query_filter.order_by(Income.date.desc())
    
    pagination = query_filter.paginate(page=query.page, per_page=query.per_page)
    
    items = []
    for income in pagination.items:
        items.append({
            "id": income.id,
            "description": income.description,
            "amount": income.amount,
            "date": income.date.isoformat(),
            "is_recurring": income.is_recurring,
            "recurrence_end_date": income.recurrence_end_date.isoformat() if income.recurrence_end_date else None,
            "category_id": income.category_id,
            "user_id": income.user_id,
            "created_at": income.created_at_str,
            "updated_at": income.updated_at_str
        })
    
    data = {
        "items": items,
        "total": pagination.total,
        "page": query.page,
        "per_page": query.per_page,
        "total_pages": pagination.pages
    }
    
    return success_response(
        message="Receitas listadas com sucesso",
        data=data,
        code=200
    )

@income_bp.get(
    "/<int:income_id>",
    responses={
        "200": SuccessResponse,
        "401": ErrorResponse,
        "404": ErrorResponse
    },
    security=[{"bearerAuth": []}]
)
@token_required
@api_response
def get_income(path: IncomeIdPath):
    """Obtém uma receita específica"""
    income = Income.query.filter_by(
        id=path.income_id,
        user_id=get_income.current_user.id
    ).first_or_404()
    
    income_data = {
        "id": income.id,
        "description": income.description,
        "amount": income.amount,
        "date": income.date.isoformat(),
        "is_recurring": income.is_recurring,
        "recurrence_end_date": income.recurrence_end_date.isoformat() if income.recurrence_end_date else None,
        "category_id": income.category_id,
        "user_id": income.user_id,
        "created_at": income.created_at_str,
        "updated_at": income.updated_at_str
    }
    
    return success_response(
        message="Receita encontrada com sucesso",
        data=income_data,
        code=200
    )

@income_bp.patch(
    "/<int:income_id>",
    responses={
        "200": SuccessResponse,
        "400": ErrorResponse,
        "401": ErrorResponse,
        "404": ErrorResponse
    },
    security=[{"bearerAuth": []}]
)
@token_required
@api_response
def update_income(path: IncomeIdPath, body: TransactionUpdate):
    """Atualiza uma receita"""
    try:
        income = Income.query.filter_by(
            id=path.income_id,
            user_id=update_income.current_user.id
        ).first_or_404()
        
        if body.category_id is not None:
            # Verifica se a nova categoria pertence ao usuário
            Category.query.filter_by(
                id=body.category_id,
                user_id=update_income.current_user.id,
                type='income'
            ).first_or_404()
            income.category_id = body.category_id
        
        if body.description is not None:
            income.description = body.description
        if body.amount is not None:
            income.amount = body.amount
        if body.date is not None:
            income.date = body.date
        if body.is_recurring is not None:
            income.is_recurring = body.is_recurring
        if body.recurrence_end_date is not None:
            income.recurrence_end_date = body.recurrence_end_date
        
        db.session.commit()
        
        income_data = {
            "id": income.id,
            "description": income.description,
            "amount": income.amount,
            "date": income.date.isoformat(),
            "is_recurring": income.is_recurring,
            "recurrence_end_date": income.recurrence_end_date.isoformat() if income.recurrence_end_date else None,
            "category_id": income.category_id,
            "user_id": income.user_id,
            "created_at": income.created_at_str,
            "updated_at": income.updated_at_str
        }
        
        return success_response(
            message="Receita atualizada com sucesso",
            data=income_data,
            code=200
        )
        
    except IntegrityError:
        db.session.rollback()
        return error_response(
            message="Erro ao atualizar receita",
            errors=[ErrorDetail(field="integrity", message="Erro de integridade no banco de dados")],
            code=400
        )

class DeleteIncomeQueryParams(BaseModel):
    delete_recurring: bool = False

@income_bp.delete(
    "/<int:income_id>",
    responses={
        "200": SuccessResponse,
        "401": ErrorResponse,
        "404": ErrorResponse
    },
    security=[{"bearerAuth": []}]
)
@token_required
@api_response
def delete_income(path: IncomeIdPath, query: DeleteIncomeQueryParams):
    """Deleta uma receita"""
    try:
        income = Income.query.filter_by(
            id=path.income_id,
            user_id=delete_income.current_user.id
        ).first_or_404()
        
        if query.delete_recurring and income.is_recurring:
            # Deleta todas as receitas recorrentes futuras
            Income.query.filter(
                Income.user_id == delete_income.current_user.id,
                Income.description == income.description,
                Income.category_id == income.category_id,
                Income.is_recurring == True,
                Income.date >= income.date
            ).delete()
        else:
            db.session.delete(income)
        
        db.session.commit()
        
        return success_response(
            message="Receita(s) deletada(s) com sucesso",
            code=200
        )
        
    except IntegrityError:
        db.session.rollback()
        return error_response(
            message="Erro ao deletar receita",
            errors=[ErrorDetail(field="integrity", message="Erro de integridade no banco de dados")],
            code=400
        )