from flask_openapi3 import OpenAPI, Info, Tag
from app.config import Config
from flask import Flask
from flask_cors import CORS
from pydantic import BaseModel

info = Info(
    title="Smart Wallet API",
    version="1.0.0",
    description="""
    API de Controle Financeiro pessoal com recursos de:
    - Autenticação JWT
    - Gerenciamento de usuários
    - Categorias de despesas e receitas
    - Controle de despesas e receitas recorrentes
    - Relatórios financeiros
    """
)

security_scheme = {
    "type": "http",
    "scheme": "bearer",
    "bearerFormat": "JWT",
    "description": "Insira o token JWT com o prefixo Bearer"
}

app = OpenAPI(
    __name__, 
    info=info
)

# Habilita CORS para todos os endpoints
CORS(app, resources={r"/api/*": {"origins": "*"}})

# Configurações de respostas e validação
app.config['REMOVE_422_RESPONSE'] = True
app.config['AUTO_VALIDATE'] = False  # Desativa validação automática do Pydantic
app.config['AUTO_STATUS_CODE'] = False  # Não adiciona códigos de status automaticamente

# Configuração de segurança
app.security_schemes = {
    "bearerAuth": {
        "type": "http",
        "scheme": "bearer",
        "bearerFormat": "JWT",
        "description": "Insira o token JWT com o prefixo Bearer. Exemplo: Bearer seu-token-aqui"
    }
}

# Define que todas as rotas são protegidas por padrão
app.security = [{"bearerAuth": []}]

# Configura CORS para aceitar o cabeçalho Authorization
app.config['CORS_HEADERS'] = 'Content-Type,Authorization'

# Força o Content-Type application/json em todas as respostas
app.config['DEFAULT_CONTENT_TYPE'] = 'application/json'

app.config.from_object(Config)

# Inicialização das extensões
from app import db, migrate
db.init_app(app)
migrate.init_app(app, db)

# Registro dos blueprints
from app.routes import user_bp, auth_bp, category_bp, expense_bp, income_bp, report_bp
app.register_api(auth_bp)
app.register_api(user_bp)
app.register_api(category_bp)
app.register_api(expense_bp)
app.register_api(income_bp)
app.register_api(report_bp)

def init_db():
    with app.app_context():
        import os
        import flask_migrate
        
        # Cria a pasta migrations se não existir
        if not os.path.exists('migrations'):
            flask_migrate.init()
            
        # Cria as tabelas se não existirem
        db.create_all()
        
        # Detecta alterações e cria uma nova migração
        flask_migrate.migrate()
        
        # Aplica as migrações
        flask_migrate.upgrade()

if __name__ == '__main__':
    init_db()  # Inicializa o banco de dados
    app.run(debug=True)
